<?php

require_once 'ControllerUtils.php';

/**
 * Skeleton subclass for representing a row from the 'SecurityAction' table.
 *
 * Actions del sistema
 *
 * @package security
 */
class SecurityAction extends BaseSecurityAction {

	const LEVEL_ALL = 1073741823;

 /**
	* Genera un objeto SecurityAction a partir de un nombre de una accion
	* @return object SecurityAction configurado
	*/
	public static function createFromAction($action) {

		$securityAction = new SecurityAction();
		$securityAction
			->setAccess(1)
			->setAccessAffiliateUser(0)
			->setAccessRegistrationUser(0)
			->setActive(1);

		$pair = ControllerUtils::getPairForAction($action);
		$securityAction->setAction($pair['withoutDo'])->setPair($pair['withDo']);

		$module = $securityAction->findModuleNameFromActionName();
		$securityModule = SecurityModuleQuery::create()->findOneByModule($module);
		if (!$securityModule)
			$securityModule = SecurityModule::createFromModule($module);

		$securityAction->setSecurityModule($securityModule);

		return $securityAction;
	}

 /**
	* Obtiene el nombre del modulo a partir del nombre de la accion
	* @return string nombre del modulo
	*/
	public function findModuleNameFromActionName() {
		return ControllerUtils::getModuleForAction($this->getAction());
	}

 /**
	* Setea el acceso segun el bitlevel
	*/
	public function setAccessForBitLevel($access, $bitLevel) {
		if ($access)
			$this->setAccess($this->getAccess() | $bitLevel);
		else
			$this->setAccess($this->getAccess() & ~$bitLevel);
	}

 /**
	* Obtiene la etiqueta de ese Action segun el idioma actual
	* @return string label la etiqueta
	*/
	public function getLabel(){
		try{
			$actionLabel = SecurityActionLabelQuery::create()->filterByAction($this->getAction())->filterByLanguage(Common::getCurrentLanguageCode())->findOne();
			if (!empty($actionLabel))
				return $actionLabel->getLabel();
			else
				return $this->getAction();
		}
		catch (PropelException $e) {}
	}

 /**
	* Obtiene la informacion de ese Action por codigo de lenguaje.
	* @return SecurityActionLabel informacion de lenguaje.
	*/
	public function getActionInfo($languageCode = NULL) {
		if ($languageCode != NULL)
			$language = $languageCode;
		else
			$language = Common::getCurrentLanguageCode();

		try{
			$actionLabel = SecurityActionLabelQuery::create()->filterByAction($this->getAction())->filterByLanguage($language)->findOne();
			return $actionLabel;
		}
		catch (PropelException $e) { return NULL; }
	}

 /**
	* Obtiene el SQL Insert Correspondiente a la tabla con los datos de la instancia
	* @return string con query para insertar los datos de seguridad en tabla correspondiente
	*/
	public function getSQLInsert() {

		return "INSERT INTO `security_action` (`action`, `module`, `section`, `access`, `accessAffiliateUser`, `active`, `pair`, `noCheckLogin`, `accessRegistrationUser`) " .
		        "VALUES ('" . $this->getAction() . "', '" . $this->getModule() . "', '" . $this->getSection() . "', '" .
		        $this->getAccess() . "', '" . $this->getAccessAffiliateUser() ."', '" . $this->getActive() . "', '" . 
		        $this->getPair() . "', '" . (!empty($this->getNoCheckLogin()) ? $this->getNoCheckLogin() : '0') . "', '" . $this->getAccessRegistrationUser() . "');";
	}

 /**
	* genera el codigo SQL de limpieza de las tablas afectadas al modulo.
	* @return string SQL
	*/
	public function getSQLCleanup($module) {
		$sql = 'DELETE FROM `security_action` WHERE `module` = ' . "'" . $module . "';\n";
		$sql .= "OPTIMIZE TABLE `security_action`;";
		return  $sql;
	}

 /**
	* Indica si el valor pasado corresponde al bitlevel actual de admin de la instancia
	* @param integer bitlevel
	* @return true en caso afirmativo, false sino.
	*/
	public function hasAccessBitLevel($bitLevel) {
		return ($this->getAccess() & $bitLevel);
	}

 /**
	* Indica si el valor pasado corresponde al bitlevel actual de afiliado de la instancia
	* @param integer bitlevel
	* @return true en caso afirmativo, false sino.
	*/
	public function hasAccessAffiliateBitLevel($bitLevel) {
		return ($this->getAccessAffiliateUser() & $bitLevel);
	}

 /**
	* Indica si hay acceso a todos en este caso de usuario general
	* @return boolean
	*/
	public function hasAllAccess() {
		return ($this->hasAccessBitLevel(1) && $this->hasAccessBitLevel(2) && $this->hasAccessBitLevel(4));
	}

 /**
	* Indica si hay acceso a todos en este caso de afiliado
	* @return boolean
	*/
	public function hasAllAffiliateAccess() {
		return ($this->hasAccessAffiliateBitLevel(1) && $this->hasAccessAffiliateBitLevel(2) && $this->hasAccessAffiliateBitLevel(4));
	}

 /**
	* Obtiene los permisos en base al tipo de usuario
	* @return int $access bit level de permisos de la accion
	*/
	public function getAccessByUser($user) {
		$userClass = get_class($user);
		$method = "getAccess";
		if ($userClass != "User")
			$method .= $userClass;
		$actionBitLevel = $this->$method();

		$level = $user->getLevel();

		if ($level->getBitLevel() & $actionBitLevel)
			return true;
		else
			return false;
	}

 /**
	* Obtiene los permisos en base al tipo de usuario
	* @return int $access bit level de permisos de la accion
	*/
	public function getAccessByUserType($user) {
		$userClass = get_class($user);
		$access = 0;
		$method = "getAccess";
		if ($userClass != "User")
			$method .= $userClass;
		$access = $this->$method();
		return $access;
	}

 /**
	* Obtiene el nivel de acceso para un usuario determinado
	* @return integet nivel de acceso
	*/
	public function getFullAccessByUser($user) {

		$noCheckLoginAction = $this->getNoCheckLogin();

		//Si es un action que no chequea login, el nivel de acceso del action es 1 en cada bit
		if ($noCheckLoginAction)
			return self::LEVEL_ALL;

		$access = $this->getAccessByUserType($user);

		//Si el access del action es 0 significa que herada los permisos de su modulo
		if ($access == 0) {
			$actualModule = $this->getSecurityModule();
			$access = $actualModule->getAccessByUserType($user);
		}
		return $access;
	}

 /**
	* Indica si un suario tiene acceso a una accion determinada
	* @return boolean  (true tiene acceso, false no lo tiene)
	*/
	public function hasAccess($user) {
		$access = $this->getFullAccessByUser($user);
		$level = $user->getLevel();

		if (empty($level) || ($level->getBitLevel() & $access) == 0)
			return false;
		else
			return true;
	}

 /**
	* Indica si una accion no verifica login
	* @return int (1 si no requiere login)
	*/
	public function getOverallNoCheckLogin() {
		$noCheckLogin = 0;
		if ($this->getActive() != 0)
			if ($this->getNoCheckLogin() == 1)
				$noCheckLogin = 1;
		else {
			$securityModule = $this->getSecurityModule();
			if ($securityModule->getNoCheckLogin() == 1)
				$noCheckLogin = 1;
		}
		return $noCheckLogin;
	}

 /**
	* Indica si una accion no verifica login
	* @return int (1 si no requiere login)
	*/
	public function getActionNoCheckLogin() {
		$noCheckLogin = 0;
		if ($this->getActive() == 1)
			if ($this->getNoCheckLogin() == 1)
				$noCheckLogin = 1;
		return $noCheckLogin;
	}

 /**
	* Obtiene un action a partir de su nombre o del par
	* @param string $action nombre del action
	* @return object $obj action encontrado
	*/
	public static function getByNameOrPair($action) {
		$securityAction = SecurityActionQuery::create()
												->setIgnoreCase('true')
												->filterByAction($action)
													->_or()
												->filterByPair($action)
												->findOne();
		return $securityAction;
	}

	/**
	 * Obtiene un array con todas las acciones a las que se tiene permiso
	 * @return array de acciones
	 */
	public static function getAccessToActions($actions) {
		$access = array();

		foreach ($actions as $action) {
			$lcAction = lcfirst($action);
			$securityAction = SecurityAction::getByNameOrPair($lcAction);
			if (!empty($securityAction)) {

				$access[$action] = array();
				$access[$action]['noCheckLogin'] = $securityAction->getNoCheckLogin();

				$bitLevel = $securityAction->getAccess();
				if ($bitLevel == SecurityModule::LEVEL_ALL) {
					$access[$action]['bitLevel'] = 0;
					$access[$action]['all'] = 1;
				}
				else
					$access[$action]['bitLevel'] = $bitLevel;
				// end User

				if (class_exists("AffiliateLevel")) {
					$bitLevelAffiliate = $securityAction->getAccessAffiliateUser();
					if ($bitLevelAffiliate == SecurityModule::LEVEL_ALL) {
						$access[$action]['bitLevelAffiliate'] = 0;
						$access[$action]['affiliateAll'] = 1;
					}
					else
						$access[$action]['bitLevelAffiliate'] = $bitLevelAffiliate;
				} // end AffiliateUser

				if (class_exists("RegistrationUser"))
					$access[$action]['permissionRegistration'] = $securityAction->getAccessRegistrationUser();
 				// end RegistrationUser

				if (class_exists("ClientUser")) {
					$bitLevelClient = $securityAction->getAccessClientUser();
					if ($bitLevelAffiliate == SecurityModule::LEVEL_ALL) {
						$access[$action]['bitLevelClient'] = 0;
						$access[$action]['clientAll'] = 1;
					}
					else
						$access[$action]['bitLevelClient'] = $bitLevelClient;
				} // end ClientUser

			}
		}
		return $access;
	}

} // SecurityAction
