<?php

use Egytca\Status\OnTheFly\Status;
use Egytca\Tablero\Utils;

/**
 * Numeric Indicator
 */
class NumericIndicator extends Indicator {

	use Status;

	const INTERVAL_DAILY = 'P1D';
	const INTERVAL_WEEKLY = 'P1W';
	const INTERVAL_MONTHLY = 'P1M';
	const INTERVAL_QUARTERLY = 'P3M';
	const INTERVAL_SEMIANNUAL = 'P6M';
	const INTERVAL_YEARLY = 'P1Y';

	protected $delayedTolerance;
	protected $ontimeTolerance;

	/**
	 * Constructs a new NumericIndicator class, setting the classKey column to IndicatorPeer::CLASSKEY_2.
	 */
	public function __construct() {
		parent::__construct();
		$this->setClasskey(IndicatorPeer::CLASSKEY_2);

		$timeTolerances = ConfigModule::get('indicators', 'valueTolerances');
		$this->delayedTolerance = $timeTolerances['delayed'];
		$this->ontimeTolerance = $timeTolerances['ontime'];
	}

	function _calculateStatus() {

		if ($this->countExpectedIndicatorValues() === 0)
			return 'undefined';

		$lastExpectedValuePastDue = $this->getLastExpectedValuePastDue();
		if (!$lastExpectedValuePastDue)
			return ('planned');

		$todayTime = strtotime('today');

		$lastRealValue = $this->getLastRealIndicatorValue();
		if (!$lastRealValue)
			return 'late';

		$lastExpectedValue = $this->getLastExpectedValue(); // existe porque el estado no es 'undefined'
		if ( $lastExpectedValue->getDate('U') < $todayTime // todas las fechas vencidas
				&& $lastRealValue->getValue() >= $lastExpectedValue->getValue() )
			return 'finished';

		$ontimeMinValue = (1 - $this->ontimeTolerance) * $lastExpectedValuePastDue->getValue();
		if ($lastRealValue->getValue() >= $ontimeMinValue)
			return 'ontime';

		$delayedMinValue = (1 - $this->delayedTolerance) * $lastExpectedValuePastDue->getValue();
		if ($lastRealValue->getValue() >= $delayedMinValue)
			return 'delayed';

		// $lastRealValue->getValue() < $delayedMinValue
		return 'late';
	}

	function getExpectedIndicatorValues($criteria = null) {
		$criteriaPlusClassKey = ExpectedIndicatorValueQuery::create(null, $criteria);
		return $this->getIndicatorValues($criteriaPlusClassKey);
	}

	function getRealIndicatorValues($criteria = null) {
		$criteriaPlusClassKey = RealIndicatorValueQuery::create(null, $criteria);
		return $this->getIndicatorValues($criteriaPlusClassKey);
	}

	function countExpectedIndicatorValues($criteria = null) {
		$criteriaPlusClassKey = ExpectedIndicatorValueQuery::create(null, $criteria);
		return $this->countIndicatorValues($criteriaPlusClassKey);
	}

	function countRealIndicatorValues($criteria = null) {
		$criteriaPlusClassKey = RealIndicatorValueQuery::create(null, $criteria);
		return $this->countIndicatorValues($criteriaPlusClassKey);
	}

	function getLastExpectedValue() {
		return ExpectedIndicatorValueQuery::create()
			->filterByNumericIndicatorId($this->getId())
			->orderByDate(Criteria::DESC)
			->findOne();
	}

	function getLastExpectedValuePastDue() {

		$today = Utils::getFormattedTodayDate();

		return ExpectedIndicatorValueQuery::create()
			->filterByNumericIndicatorId($this->getId())
			->filterByDate($today, Criteria::LESS_THAN)
			->orderByDate(Criteria::DESC)
			->findOne();
	}

	function getLastRealIndicatorValue() {
		return RealIndicatorValueQuery::create()
			->filterByNumericIndicatorId($this->getId())
			->orderByDate(Criteria::DESC)
			->findOne();
	}

	function getLastRealIndicatorValuePriorToToday() {

		$today = Utils::getFormattedTodayDate();

		return RealIndicatorValueQuery::create()
			->filterByNumericIndicatorId($this->getId())
			->filterByDate($today, Criteria::LESS_THAN)
			->orderByDate(Criteria::DESC)
			->findOne();
	}

	function autocreateExpectedIndicatorValues($startDateString, $endDateString, $startValue, $endValue, $interval) {

		if ($this->countExpectedIndicatorValues() > 0)
			throw new Exception('numericIndicator must not have any expectedIndicatorValues');

		global $system;
		$dateFormat = $system['config']['system']['parameters']['dateFormat']['value'];

		$startDate = DateTime::createFromFormat($dateFormat, $startDateString);
		$endDate = DateTime::createFromFormat($dateFormat, $endDateString);

		$dates = new DatePeriod(
			$startDate,
			new DateInterval($interval),
			$endDate
		);

		$createExpectedIndicatorValue = function($date, $fn) {

			$expectedIndicatorValue = new ExpectedIndicatorValue();
			$expectedIndicatorValue->setNumericIndicatorId($this->getId())
				->setDate($date)
				->setValue($fn($date))
				->save();

			return $expectedIndicatorValue;
		};

		$createLinear = function($x0, $x1, $y0, $y1) {
			return function($x) use ($x0, $x1, $y0, $y1) {
				$a = ($y1 - $y0) / ($x1 - $x0);
				$b = $y0;
				$deltaX = $x - $x0;
				return $a *  $deltaX + $b;
			};
		};

		$aux = $createLinear(
			$startDate->getTimestamp(), $endDate->getTimestamp(),
			$startValue, $endValue
		);
		$linear = function($date) use ($aux) {
			return $aux($date->getTimestamp());
		};

		$expectedIndicatorValues = new PropelObjectCollection();
		foreach ($dates as $date) {
			$expectedIndicatorValues->append($createExpectedIndicatorValue($date, $linear));
		}
		$expectedIndicatorValues->append($createExpectedIndicatorValue($endDate, $linear));

		return $expectedIndicatorValues;
	}

} // NumericIndicator
