<?php
/**
 * Clase ModuleVerify.
 * Contiene metodos para verificar integridad de los sistemas comparando y actualizando
 * hash de directorios y del sistema
 *
 * @package  common
 */

namespace Egytca;

class ModuleVerify{

	/** Directorio donde se guardan los hashs generados y serializados */
	private $fingerprintsDir = "./WEB-INF/fingerprints/";

	function __construct($module){

		$dirs = $this->getDirs();
		$this->dir = $dirs[$module];

		$this->ignores = $this->getIgnores();
		$this->extIgnores = $this->getExtIgnores();

		// si el directorio de fingerprints no existe lo crea
		if (!file_exists($this->fingerprintsDir))
			mkdir($this->fingerprintsDir, 0755, true);

		$file = str_replace('./','',$dirs[$module]['dir']);
		$this->file = $this->fingerprintsDir . str_replace('/','.',$file);

		// obtengo los hashes actuales
		$this->hashes = unserialize(file_get_contents($this->file));

		// si no existe el archivo o no hay ningun hash creo el arreglo
		if (!$this->hashes || !is_array($this->hashes))
			$this->hashes = array();

		// array para los archivos nuevos
		$this->newFiles = array();

		// array para los archivos que cambiaron
		$this->changedFiles = array();

	}

	/**
	 * Determina si hay permisos de escritura en el directorio de fingerprints
	 *
	 * @return bool true si puede escribir, false si no
	 */
	public static function hasWriteAccess() {

		$hasWriteAccess = true;
		if (!file_exists($this->$fingerprintsDir)) {
			$created = mkdir($this->$fingerprintsDir, 0755, true);
			if (!$created)
				$hasWriteAccess = false;
		else
			if (!Common::isWritable($this->$fingerprintsDir))
				$hasWriteAccess = false;
		}
		return $hasWriteAccess;
	}

	/**
	 * Obtiene el listado de directorios a verificar
	 *
	 * @return array con  listado de directorios a verificar
	 */
	public static function getDirs() {
		// cada elemento tiene un nombre como clave, id para el form, path y si solo verifica archivos
		$dirs = array(
			'config' => array('dir' => './config',
												'onlyFiles' => false,
												'hash' => md5_file('./WEB-INF/fingerprints/config')
												),
			'classes' => array('dir' => './WEB-INF/classes',
												 'onlyFiles' => true,
												 'hash' => md5_file('./WEB-INF/fingerprints/WEB-INF.classes')
												 ),
			'includes' => array('dir' => './WEB-INF/classes/includes',
													'onlyFiles' => false,
													'hash' => md5_file('./WEB-INF/fingerprints/WEB-INF.classes.includes')
													),
			'lib-phpmvc' => array('dir' => './WEB-INF/lib-phpmvc',
													'onlyFiles' => false,
													'hash' => md5_file('./WEB-INF/fingerprints/WEB-INF.lib-phpmvc')
													),
			'propel' => array('dir' => './WEB-INF/propel',
												'onlyFiles' => true,
												'hash' => md5_file('./WEB-INF/fingerprints/WEB-INF.propel')
												)
		);

		// obtengo todos los modulos
		if ($handle = opendir('./WEB-INF/classes/modules')) {
			$ignore = array('.', '..','.svn','.git');
			while (false !== ($dirName = readdir($handle)))
				if (!in_array($dirName, $ignore))
					$dirs[$dirName] = array('dir' => "./WEB-INF/classes/modules/$dirName",
																	'onlyFiles' => false,
																	'hash' => md5_file("./WEB-INF/fingerprints/WEB-INF.classes.modules.$dirName")
																	);
			closedir($handle);
		}
		return $dirs;
	}

	/**
	 * Obtiene el hash del directorio obtenido a partir del archivo de fingerprint correspondiente
	 *
	 * @return string hash de verificacion de un directorio
	 */
	public function getDirectoryHash() {
		return md5_file($this->file);
	}

	/**
	 * Obtiene el hash del directorio obtenido a partir del archivo de fingerprint correspondiente
	 *
	 * @return string el hash del directorio
	 */
	public function getNewHash() {
		return md5(serialize($this->hashes));
	}

	/**
	 * Obtiene el listado de archivos y directorios a ignorar
	 *
	 * @return array con el listado de archivos y directorios a ignorar
	 */
	public static function getIgnores() {
		return array('.','..','application-conf.php','application-classmap.php','fingerprints','migrations','sql','.svn');
	}

	/**
	 * Obtiene el listado de extensiones a ignorar
	 *
	 * @return array con el listado de extensiones a ignorar
	 */
	public static function getExtIgnores() {
		return array('svn','bak','sql','data');
	}

	/**
	 * Obtiene el hash de un directorio
	 *
	 * @return string con el hash de un directorio o false si no lo puede abrir
	 */
	public function lookFingerprintsDir(){
		$path = "./WEB-INF/fingerprints";
		$handle = @opendir($path);
		if (!$handle)
			return false;

		$ignores = ModuleVerify::getIgnores();
		$extIgnores = ModuleVerify::getExtIgnores();

		$allHashes = "";

		while ($item = readdir($handle)) {
			$ext = pathinfo($item, PATHINFO_EXTENSION);
			if (!in_array($item, $ignores) && !in_array($ext, $extIgnores)) {
				// unserialize en un string
				$allHashes.= file_get_contents($path."/".$item);
			}
		}
		closedir($handle);
		return md5($allHashes);
	}

	/**
	 * Comprueba un directorio, si puede abrirlo obtiene sus archivos, si puede abrirlos genera los hash
	 * Si no duvuelve false
	 *
	 * @return bool 
	 */
	public function lookDir($path) {
		$handle = @opendir($path);
		if (!$handle)
			return false;

		while ($item = readdir($handle)) {
			$this->checkItem($path, $item);
			
		}
		closedir($handle);
		return true;
	}
	
	/**
	 * Comprueba un item (directorio o archivo), si es directorio sigue buscando adentro, si es archivo verifica los hashes
	 * 
	 * @return void 
	 */
	private function checkItem($path, $item){
		
		$ext = pathinfo($item, PATHINFO_EXTENSION);
		if (!in_array($item, $this->ignores) && !in_array($ext, $this->extIgnores)) {
			if (is_dir($path."/".$item) && !$this->dir['onlyFiles'])
				$this->lookDir($path."/".$item);
			else
				$this->checkFile($path."/".$item);
		}
	} 

	/**
	 * Comprueba un archivo, si existe genera su hash
	 *
	 * @return void - modifica el arreglo de hashes
	 */
	private function checkFile($file) {
		if(!file_exists($file))
			unset($this->hashes[$file]);
		else {
			if (is_readable($file)){
				 //echo "readable";
				if (!isset($this->hashes[$file])) {
					 // archivo nuevo
					 $this->hashes[$file] =  md5_file($file);
					 $this->newFiles[$file] = $this->hashes[$file];
				}
				elseif ($this->hashes[$file] != md5_file($file)) {
					 $this->hashes[$file] =  md5_file($file);
					 $this->changedFiles[$file] = $this->hashes[$file];
				}
				else {
					 $this->hashes[$file] =  md5_file($file);
				}
			}
		}
	}

} // ModuleVerify
