<?php

/**
 * Formats a Position object using wildcards.
 * If $position is null, returns an empty string.
 *
 * Available wildcards are:
 *     %p: position name
 *     %t: active tenure name
 *
 *
 * Conditional formatting can be configured using the following format:
 *
 *     [(<conditional text>)%<wildcard>?]
 *
 *     where <wildcard> will be used as a condition;
 *     if <wildcard> is empty, the conditional string will not be shown.
 *
 * Example:
 *     Formatting a position "Slave" with an active tenure "Diego"
 *     and a position "Free programmer" with no active tenure,
 *     with the following format:
 *
 *     %p[( (%t))%t?]
 *
 *     will print
 *     "Slave (Diego)"
 *     and
 *     "Free programmer" // no parenthesis, since there is no active tenure
 */
function smarty_modifier_position($position,
		$format = '%p[( - %t)%t?]') {

	$wildcards = ['p', 't'];

	if (!$position) return '';

	$p = $position->getName();

	$activeTenureName = $position->getActiveTenureName();
	if ($activeTenureName) {
		$t = $activeTenureName;
	} else {
		$t = '';
	}

	$adjustedFormat = $format;
	foreach ($wildcards as $wildcard) {
		$regexp = "/\\[\\(([^\\]]+)\\)%$wildcard\\?]/";
		$replacement = $$wildcard ? '$1' : '';
		$adjustedFormat = preg_replace($regexp, $replacement, $adjustedFormat);
	}

	return preg_replace(["/%p/", "/%t/"], [$p, $t], $adjustedFormat);
}
