<?php

use Egytca\ErrorReporting\ErrorReport;

/**
 * BasedoDeleteAction
 *
 * Meta clase para eliminar un objeto de la base de datos
 *
 * @package phpMVC
 * @author Modulos Empresarios / Egytca
 * @copyright Copyright (c) 2015, Egytca
 */

class BaseDoDeleteAction extends BaseAction {

	private $entityClassName;
	protected $smarty;
	protected $entity;
	protected $ajaxTemplate;
	protected $params;
	protected $filters;
	protected $actionLog = true;
	protected $page;
	protected $forwardName = 'success';
	protected $forwardFailureName = 'failure';

	/**
	 * Constructor
	 *
	 * @param string $entityClassName Nombre de la clase de la entidad que se va a eliminar
	 */
	function __construct($entityClassName) {
		if (empty($entityClassName))
			throw new Exception('$entityClassName must be set');
		$this->entityClassName = $entityClassName;
		if (substr(get_class($this), -7, 1) != 'X')
			$this->ajaxTemplate = str_replace('Action', '', get_class($this)) . 'X.tpl';
		else
			$this->ajaxTemplate = str_replace('Action', '', get_class($this)) . '.tpl';
	}

	/**
	 * Execute
	 * @see BaseAction::execute()
	 */
	function execute($mapping, $form, &$request, &$response) {

		parent::execute($mapping, $form, $request, $response);

		// los listados borran usando un GET!!
		// y esta warning jamas se va a ver porque despues del delete se redirecciona :(
		trigger_error('GET requests should not be allowed', E_USER_WARNING);

		// Acciones a ejecutar antes de eliminar el objeto
		if ($this->preDelete() === false)
			return $this->addParamsAndFiltersToForwards($params, $this->filters, $mapping, $this->forwardFailureName);

		$id = $request->getParameter('id');
		if (!empty($id)) {

			$this->entity = BaseQuery::create($this->entityClassName)->findOneById($id);

			if (is_null($this->entity)) {
				//Elijo la vista basado en si es o no un pedido por AJAX
				if ($this->isAjax()) {
					throw new Exception(); // Buscar una mejor forma de que falle AJAX
				} else {
					$this->smarty->assign('notValidId', 'true');
					return $this->addParamsAndFiltersToForwards($this->params, $this->filters, $mapping, $this->forwardName);
				}
			}

			if (method_exists($this->entity, 'isDeletable') && !$this->entity->isDeletable())
				return $this->addParamsAndFiltersToForwards($this->params, $this->filters, $mapping, $this->forwardFailureName);

			try {

				$this->entity->delete();
				$action = 'delete';
				$logSufix = ', ' . Common::getTranslation('action: '.$action, 'common');
				if ($this->actionLog && method_exists($this->entity, 'getLogData'))
					Common::doLog('success', $this->entity->getLogData() . $logSufix);

				// Acciones a ejecutar despues de eliminar el objeto
				return $this->addParamsAndFiltersToForwards($this->params, $this->filters, $mapping, $this->forwardName);
			} catch (Exception $e) {

				$this->errors->add($e);

				global $system;
				if (($system['config']['system']['parameters']['debugMode']['value'] == 'YES')) {
					ErrorReport::createFromException($e)->sendByMail();
				}

				return $this->addParamsAndFiltersToForwards($this->params, $this->filters, $mapping, $this->forwardFailureName);
			}

			$this->postDelete();

		}

		return $this->addParamsAndFiltersToForwards($this->params, $this->filters, $mapping, $this->forwardFailureName);
	}

	/**
	 * preDelete
	 * Acciones a tomar antes de eliminar el objeto
	 */
	protected function preDelete() {

		// Procesar los valores del request asociados a filtros
		$this->prepareFilters();

	}

	/**
	 * postDelete
	 * Acciones a tomar despues de eliminar el objeto
	 */
	protected function postDelete() {
		// default: do nothing
	}

}
